/* globals dwbrackets */
define(function (require, exports) {
    "use strict";

    var CommandManager    = brackets.getModule("command/CommandManager"),
        FileSystem        = brackets.getModule("filesystem/FileSystem"),
        PreferencesManager  = brackets.getModule("preferences/PreferencesManager"),
        StringUtils        = brackets.getModule("utils/StringUtils");

    var Panel             = require("src/Panel"),
        DWGitEvents       = require("src/DWGit/DWGitEvents"),
        Git               = require("src/git/Git"),
        ErrorHandler      = require("src/ErrorHandler"),
        Preferences       = require("src/Preferences"),
        HistoryViewer     = require("src/HistoryViewer"),
        History           = require("src/History"),
        Branch            = require("src/Branch"),
        NoRepo            = require("src/NoRepo"),
        Strings           = require("strings"),
        Events            = require("src/Events"),
        EventEmitter      = require("src/EventEmitter"),
        URI               = require("URI");

    var DYNAMIC_GIT_FONT_FAMILY_ID = "brackets-git-app-font";

    var curSelectedFilePath = "";
    var currentModalDialog = null;
    var isGitHistoryShown = false;

    function sendGitStatusToDreamweaver(folderPath, isGitRepo, statusFiles) {
        var responseObj = {root:folderPath, isGitRepo:false, status:"", branches:[], currentBranch:""};
        if (isGitRepo === true) {
            responseObj.isGitRepo = true;
            responseObj.status = statusFiles;
            Git.getBranches().catch(function (err) {
                dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_STATUS, JSON.stringify(responseObj));
                //ErrorHandler.showError(err, "Getting branch list failed");
            }).then(function (listBranches) {
                for(var i=0; i<listBranches.length; ++i) {
                    var branchObj = listBranches[i];
                    if(!branchObj.remote) {
                        responseObj.branches.push(branchObj.name);
                    }
                    if(branchObj.currentBranch) {
                        responseObj.currentBranch = branchObj.name;
                    }
                }
                Panel.getPanel().find(".git-history-toggle, .tags").prop("disabled", (listBranches.length === 0));
                dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_STATUS, JSON.stringify(responseObj));
            });
        } else {
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_STATUS, JSON.stringify(responseObj));
        }
    }

    function strEndsWith(subjectString, searchString, position) {
        if (position === undefined || position > subjectString.length) {
            position = subjectString.length;
        }
        position -= searchString.length;
        var lastIndex = subjectString.indexOf(searchString, position);
        return lastIndex !== -1 && lastIndex === position;
    }

    function dwSetGitRoot(gitRoot) {
        Panel.getPanel().find("#git-search-files").blur().val('');
        HistoryViewer.hide();
        curSelectedFilePath = "";
        if(gitRoot.length === 0) {
            Preferences.set("currentGitRoot", "");
            Panel.toggle(false);
            Panel.disable("not-repo");
            return;
        }

        if (strEndsWith(gitRoot, "/")) {
            gitRoot = gitRoot.slice(0, -1);
        }
        if (typeof brackets !== "undefined" && brackets.fs && brackets.fs.stat) {
            brackets.fs.stat(gitRoot + "/.git", function (err, result) {
                var exists = err ? false : (result.isFile() || result.isDirectory());
                if (exists) {
                    var prevGitRoot = Preferences.get("currentGitRoot");
                    Preferences.set("currentGitRoot", gitRoot + "/");
                    Panel.enable();
                    Git.getMergeMode().then(function (mergeInfo) {
                        EventEmitter.emit(Events.REBASE_MERGE_MODE, mergeInfo.rebaseMode, mergeInfo.mergeMode);
                    });
                    if(prevGitRoot !== Preferences.get("currentGitRoot")) {
                        Panel.disableControls(true);
                    }
                    Panel.toggle(true);
                } else {
                    Preferences.set("currentGitRoot", "");
                    Panel.toggle(false);
                    Panel.disable("not-repo");
                    sendGitStatusToDreamweaver(gitRoot, false, null);
                }
            });
            return;
        }

        FileSystem.resolve(gitRoot + "/.git", function (err, item, stat) {
            var exists = err ? false : (stat.isFile || stat.isDirectory);
            if (exists) {
                var prevGitRoot = Preferences.get("currentGitRoot");
                Preferences.set("currentGitRoot", gitRoot + "/");
                Panel.enable();
                Git.getMergeMode().then(function (mergeInfo) {
                    EventEmitter.emit(Events.REBASE_MERGE_MODE, mergeInfo.rebaseMode, mergeInfo.mergeMode);
                });
                if(prevGitRoot !== Preferences.get("currentGitRoot")) {
                    Panel.disableControls(true);
                }
                Panel.toggle(true);
            } else {
                Preferences.set("currentGitRoot", "");
                Panel.toggle(false);
                Panel.disable("not-repo");
                sendGitStatusToDreamweaver(gitRoot, false, null);
            }
        });
    }

    function dwSetDialogGitRoot(gitRoot) {
        if(gitRoot.length === 0) {
            Preferences.set("currentGitRoot", "");
            return;
        }

        if (!strEndsWith(gitRoot, "/")) {
            gitRoot = gitRoot + "/";
        }
        Preferences.set("currentGitRoot", gitRoot);
    }

    function dwRefreshPanel(bUpdateMergeStatus) {
        Panel.refresh();
        if(bUpdateMergeStatus === "true") {
            Git.getMergeMode().then(function (mergeInfo) {
                EventEmitter.emit(Events.REBASE_MERGE_MODE, mergeInfo.rebaseMode, mergeInfo.mergeMode);
            });
        }
    }

    function sendSingleFileGitStatusToDreamweaver(results, singleFilePath) {
        if(singleFilePath && !strEndsWith(singleFilePath, "/")) {
            var i,
                status = [],
                statusStr;
            for(i = 0; i < results.length; ++i) {
                if(results[i].file === singleFilePath) {
                    status = results[i].status;
                    break;
                }
            }
            if(status.length === 0 ) {
                status.push("UNMODIFIED");
            }
            if(status[0] !== "STAGED") {
                statusStr = " " + status[0];
            } else {
                statusStr = status[1];
            }
            var fullPath = Preferences.get("currentGitRoot") + singleFilePath;
            var responseString = fullPath + "\n" + statusStr;
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_FILE_STATUS, responseString);
        } else {
            sendGitStatusToDreamweaver(Preferences.get("currentGitRoot"), true, results);
        }
    }

    // after add to stage or reset ,send git file status to dreamweaver.
    //function will keep dw gitstatusmap in sync with current git status even if errors are thrown during add or reset git operation.
    function toggleStagingCallback(selList) {
        if(selList.length === 1 && (selList[0].lastIndexOf("/") !== selList[0].length - 1)) {
            Git.status(selList[0]);
        } else {
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_REFRESH_GUTTERMARKS, "");
            Git.status();
        }
    }

    function dwStage(gitRoot, selList) {
        if(selList.length > 0) {
            Git.stage(selList,false).then(function () {
                toggleStagingCallback(selList);
            }).catch(function() {
                toggleStagingCallback(selList);
            });
        }
    }

    function dwUnstage(gitRoot, selList) {
        if(selList.length > 0) {
            Git.dwUnstage(selList).then(function () {
                toggleStagingCallback(selList);
            }).catch(function(){
                toggleStagingCallback(selList);
            });
        }
    }

    function dwDiff(gitRoot, filePath) {
        if(filePath.length > 0) {
            Panel.dwShowGitDiff(filePath[0]);
        }
    }

    function dwFileHistory(gitRoot, filePath) {
        if(isGitHistoryShown) {
            return;
        }
        if (filePath && filePath.length > 0) {
            curSelectedFilePath = filePath[0];
            EventEmitter.emit(Events.HISTORY_SHOW, "FILE");
        } else {
            EventEmitter.emit(Events.HISTORY_SHOW, "GLOBAL");
        }
    }

    function dwRepoHistory(gitRoot) {
        if(isGitHistoryShown) {
            return;
        }
        EventEmitter.emit(Events.HISTORY_SHOW, "GLOBAL");
    }

    function dwCommit(gitRoot) {
        Panel.dwHandleGitCommit(false);
    }

    function dwCommitMerge(gitRoot) {
        Panel.dwHandleGitCommit(true);
    }

    function getCurrentSelectedFilePath() {
        return curSelectedFilePath;
    }

    function dwChangeGitBranch(branchName) {
        Git.checkout(branchName).then(function () {
            Panel.refresh();
        }).catch(function (err) {
            Panel.refresh();
            ErrorHandler.showError(err, Strings.ERROR_BRANCH_SWITCH);
        });
    }

    function dwOpenGitTerminal() {
        EventEmitter.emit(Events.TERMINAL_OPEN, true);
    }

    function dwCreateNewBranch() {
        EventEmitter.emit(Events.DW_CREATE_NEW_BRANCH);
    }
    
    function dwMergeBranch(srcBranch) {
        EventEmitter.emit(Events.DW_MERGE_BRANCH, srcBranch);
    }

    function dwDeleteBranch(branchName) {
        EventEmitter.emit(Events.DW_DELETE_BRANCH, branchName);
    }

    function dwCancelDialog() {
        if(currentModalDialog) {
            currentModalDialog.close();
            currentModalDialog = null;
        }
        dwCancelHistoryView();
    }

    function dwCancelHistoryView() {
        if(isGitHistoryShown) {
            HistoryViewer.hide();
            Panel.getPanel().find("#git-history-list").remove();
            $(window).off("keydown", History.keyDownHook);
            isGitHistoryShown = false;
        }
    }

    function gitDialogShown(dialogType, dialog, fileName) {
        if (window.dwGitDialogInstance !== true) {
            return;
        }
        //Cancel any dialog shown already
        dwCancelDialog();
        dwCancelHistoryView();
        var $dialog = dialog.getElement();
        //get current dialog dimensions and send a notification to DW
        var strDimensions = $dialog.width() + " " + $dialog.height();
        var params = dialogType + "\n" + strDimensions;
        if(fileName && fileName.length > 0) {
            params = params + "\n" + fileName;
        }
        dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_DIALOG_SHOWN, params);
        dialog.done(function (buttonId) {
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_DIALOG_CLOSED, "");
        });
        currentModalDialog = dialog;
    }

    function closeHistoryDialog() {
        dwCancelHistoryView();
        dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_DIALOG_CLOSED, "");
    }

    function gitHistoryViewShown(file) {
        if (window.dwGitDialogInstance !== true) {
            return;
        }
        if(isGitHistoryShown) {
            return;
        }
        //set dimensions for history view.(1000 * 700)
        var strDimensions = 1000 + " " + 700;
        var params = file ? "fileHistory" : "repoHistory";
        params = params + "\n" + strDimensions;
        if(file && file.relative && file.relative.length > 0) {
            params = params + "\n" + file.relative;
        }
        dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_DIALOG_SHOWN, params);
        $(window).on("keydown", History.keyDownHook);
        isGitHistoryShown = true;
    }

    function updateDreamweaver(reloadDocuments, refreshGutterMarks) {
        var updateArgument = "panel";
        if (reloadDocuments === true) {
            updateArgument = "docs";
        } else if (refreshGutterMarks === true) {
            updateArgument = "gutters";
        }
        dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_UPDATE_DW, updateArgument);
    }

    function dwStageAll(gitRoot) {
        var selList = [];
        Git.stageAll().then(function () {
            toggleStagingCallback(selList);
        }).catch(function() {
            toggleStagingCallback(selList);
        });
    }

    function dwUnstageAll(gitRoot) {
        var selList = [];
        Git.resetIndex().then(function () {
            toggleStagingCallback(selList);
        }).catch(function() {
            toggleStagingCallback(selList);
        });
    }

    function dwInitializeGit(gitRoot) {
        if (!strEndsWith(gitRoot, "/")) {
            gitRoot = gitRoot + "/";
        }
        EventEmitter.emit(Events.HANDLE_GIT_INIT, gitRoot);
    }

    function dwShowError(errorString) {
        ErrorHandler.showErrorDW(errorString);
    }

    function onCreateGitRepo(gitRoot, username, password) {
        dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_REPO_CREATED, gitRoot);
        if(username && username.length > 0) {
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_STORE_REMOTE_CREDENTIALS, gitRoot + "\n" + "origin" + "\n" + username + "\n" + password);
        }
    }

    function dwUpdateFileStatus(filePath) {
        Panel.refresh(filePath);
    }

    function dwReloadPreferences() {
        PreferencesManager.fileChanged(PreferencesManager.getUserPrefFile());
    }

    function dwRevertFile(gitRoot, fileList) {
        if(fileList.length > 0) {
            var filePath = fileList[0];
            Git.discardFileChanges(filePath).then(function () {
                Panel.refresh(filePath);
            }).catch(function (err) {
                ErrorHandler.showError(err, Strings.ERROR_DISCARD_CHANGES_TO_FILE);
            });
        }
    }

    function dwFetchRemote(remoteName, username, password) {
        EventEmitter.emit(Events.HANDLE_FETCH, remoteName, username, password);
    }

    EventEmitter.on(Events.FETCH_COMPLETE, function () {
        updateDreamweaver(false, false);
    });

    function dwPullFromRemote(remoteName, username, password) {
        EventEmitter.emit(Events.HANDLE_PULL, remoteName, username, password);
    }

    function dwPushToRemote(remoteName, username, password) {
        EventEmitter.emit(Events.HANDLE_PUSH, remoteName, username, password);
    }

    function dwResetLocalRepo() {
        Git.discardAllChanges().catch(function (err) {
            ErrorHandler.showError(err, Strings.ERROR_DISCARD_ALL_CHANGES);
        }).then(function () {
            updateDreamweaver(true);
        });
    }

    function dwCloneRepo(siteRoot, remoteUrl, username, password, bRemember) {
        EventEmitter.emit(Events.HANDLE_GIT_CLONE, siteRoot, remoteUrl, username, password, (bRemember === "true" ? true : false));
    }

    function dwCreateTag(tagName) {
        Git.setTagName(tagName).then(function () {
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_HISTORY, "GLOBAL");
        }).catch(function (err) {
            ErrorHandler.showError(err, Strings.ERROR_CREATE_TAG);
        });
    }

    function onManageRemotes()
    {
        Git.getRemotes().then(function (remotes) {
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_MANAGE_REMOTES, JSON.stringify(remotes));
        }).catch(function (err) {
            ErrorHandler.showError(err, Strings.ERROR_REMOTE_GET_DETAILS);
        });
    }

    function dwAddRemote(remoteName, url, username, password)
    {
        Git.createRemote(remoteName, url).then(function () {
            if(username.length > 0 && password.length > 0) {
                dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_STORE_REMOTE_CREDENTIALS, Preferences.get("currentGitRoot") + "\n" + remoteName + "\n" + username + "\n" + password);
            }
            EventEmitter.emit(Events.DW_REFRESH_REMOTES);
            Git.getRemotes().then(function (remotes) {
                dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_REMOTES_UPDATE, JSON.stringify(remotes));
            });
        }).catch(function (err) {
            ErrorHandler.showError(err, Strings.ERROR_REMOTE_ADD);
        });
    }

    function dwDeleteRemote(remoteName)
    {
        Git.deleteRemote(remoteName).then(function () {
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_REMOVE_REMOTE_CREDENTAILS,  Preferences.get("currentGitRoot") + "\n" + remoteName);
            EventEmitter.emit(Events.DW_REFRESH_REMOTES);
            Git.getRemotes().then(function (remotes) {
                dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_REMOTES_UPDATE, JSON.stringify(remotes));
            });
        }).catch(function (err) {
            ErrorHandler.showError(err, Strings.ERROR_REMOTE_DELETE);
        });
    }

    function dwSetUserDetails(name, email) {
        Git.setConfig("user.name", name, true).catch(function (err) {
            ErrorHandler.showError(err, Strings.ERROR_CHANGE_USERNAME);
        }).then(function () {
            Git.setConfig("user.email", email, true).catch(function (err) {
                ErrorHandler.showError(err, Strings.ERROR_CHANGE_USEREMAIL);
            });
        });
    }

    function dwSiteDeleted(siteRoot, siteReference) {
        Git.getRemoteNames(siteRoot).then(function (remoteNames) {
            if (remoteNames && remoteNames.length > 0) {
                var args = siteReference + "\n";
                for(var i=0;i<remoteNames.length;++i) {
                    args += remoteNames[i] + "\n";
                }
                dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_REMOVE_SITE_REMOTES_CREDENTIALS, args);
            }
        }).catch(function (err) {
            //Discard the error;
        });
    }

    function dwUpdateTheme(themeName) {
        if (themeName === "dark-theme") {
            $("body").addClass("dark");
        } else if (themeName === "light-theme") {
            $("body").removeClass("dark");
        }
    }

    function loadAppTheme() {
        var appTheme = Preferences.get("appTheme");
        dwUpdateTheme(appTheme);
    }

    function logEvent(eventString, initiatedFromFilesPanel)
    {
        var subCategory = (initiatedFromFilesPanel === true) ? "Git-FilesPanel" : "Git-Panel";
        dwbrackets.dwHeadlightLogging("Git", subCategory, eventString);
    }

    function _addDynamicAppFont(fontFamily, fontSize) {
        var propertyID = DYNAMIC_GIT_FONT_FAMILY_ID;
        var cssRule  = ".git-font";
        var $style   = $("<style type='text/css'></style>").attr("id", propertyID);
        var styleStr = StringUtils.format("{0}: {1} {2};", "font-family", fontFamily," !important");
        styleStr = styleStr + StringUtils.format("{0}: {1} {2};", "font-size", fontSize," !important");
        var lineHeight = (parseInt(fontSize, 10) + 4) + "px";
        styleStr = styleStr + StringUtils.format("{0}: {1} {2};", "line-height", lineHeight," !important");
        $style.html(cssRule + "{ " + styleStr +  " }");

        $("#" + propertyID).remove();// Let's make sure we remove the already existing item from the DOM.
        $("head").append($style);
    }

    function dwTestRemoteConnection(remoteURL, username, password) {
        var remoteUrlNew = remoteURL;
        if (remoteURL.match(/^https?:/)) {
            var uri = new URI(remoteURL);
            uri.username(username);
            uri.password(password);
            remoteUrlNew = uri.toString();
        }

        Git.testRemote(remoteUrlNew).then(function() {
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_CONNECTED_TO_REMOTE, "");
        }).catch(function (err) {
            ErrorHandler.showError(err, Strings.CONNCECTION_FAILED);
        });
    }

    function init() {
        if (window.dwGitPanelInstance === true) {
            CommandManager.register(DWGitEvents.SET_GIT_ROOT, DWGitEvents.SET_GIT_ROOT, dwSetGitRoot);
            CommandManager.register(DWGitEvents.REFRESH_PANEL, DWGitEvents.REFRESH_PANEL, dwRefreshPanel);
            CommandManager.register(DWGitEvents.STAGE, DWGitEvents.STAGE, dwStage);
            CommandManager.register(DWGitEvents.UNSTAGE, DWGitEvents.UNSTAGE, dwUnstage);
            CommandManager.register(DWGitEvents.UPDATE_FILE_STATUS, DWGitEvents.UPDATE_FILE_STATUS, dwUpdateFileStatus);
            CommandManager.register(DWGitEvents.CHANGE_BRANCH, DWGitEvents.CHANGE_BRANCH, dwChangeGitBranch);
            CommandManager.register(DWGitEvents.OPEN_TERMINAL, DWGitEvents.OPEN_TERMINAL, dwOpenGitTerminal);
            CommandManager.register(DWGitEvents.STAGE_ALL, DWGitEvents.STAGE_ALL, dwStageAll);
            CommandManager.register(DWGitEvents.UNSTAGE_ALL, DWGitEvents.UNSTAGE_ALL, dwUnstageAll);
            CommandManager.register(DWGitEvents.INITIALIZE_GIT, DWGitEvents.INITIALIZE_GIT, dwInitializeGit);
            CommandManager.register(DWGitEvents.REVERT_FILE, DWGitEvents.REVERT_FILE, dwRevertFile);
            CommandManager.register(DWGitEvents.ADD_REMOTE, DWGitEvents.ADD_REMOTE, dwAddRemote);
            CommandManager.register(DWGitEvents.DELETE_REMOTE, DWGitEvents.DELETE_REMOTE, dwDeleteRemote);
            CommandManager.register(DWGitEvents.SET_USER_DETAILS, DWGitEvents.SET_USER_DETAILS, dwSetUserDetails);
            CommandManager.register(DWGitEvents.SITE_DELETED, DWGitEvents.SITE_DELETED, dwSiteDeleted);
        } else {
            CommandManager.register(DWGitEvents.SET_GIT_ROOT, DWGitEvents.SET_GIT_ROOT, dwSetDialogGitRoot);
            CommandManager.register(DWGitEvents.DIFF, DWGitEvents.DIFF, dwDiff);
            CommandManager.register(DWGitEvents.CANCEL_DIALOG, DWGitEvents.CANCEL_DIALOG, dwCancelDialog);
            CommandManager.register(DWGitEvents.COMMIT, DWGitEvents.COMMIT, dwCommit);
            CommandManager.register(DWGitEvents.COMMIT_MERGE, DWGitEvents.COMMIT_MERGE, dwCommitMerge);
            CommandManager.register(DWGitEvents.SHOW_ERROR, DWGitEvents.SHOW_ERROR, dwShowError);
            CommandManager.register(DWGitEvents.CREATE_NEW_BRANCH, DWGitEvents.CREATE_NEW_BRANCH, dwCreateNewBranch);
            CommandManager.register(DWGitEvents.MERGE_BRANCH, DWGitEvents.MERGE_BRANCH, dwMergeBranch);
            CommandManager.register(DWGitEvents.DELETE_BRANCH, DWGitEvents.DELETE_BRANCH, dwDeleteBranch);
            CommandManager.register(DWGitEvents.FETCH_REMOTE, DWGitEvents.FETCH_REMOTE, dwFetchRemote);
            CommandManager.register(DWGitEvents.PULL_FROM_REMOTE, DWGitEvents.PULL_FROM_REMOTE, dwPullFromRemote);
            CommandManager.register(DWGitEvents.PUSH_TO_REMOTE, DWGitEvents.PUSH_TO_REMOTE, dwPushToRemote);
            CommandManager.register(DWGitEvents.FILE_HISTORY, DWGitEvents.FILE_HISTORY, dwFileHistory);
            CommandManager.register(DWGitEvents.REPO_HISTORY, DWGitEvents.REPO_HISTORY, dwRepoHistory);
            CommandManager.register(DWGitEvents.RESET_LOCAL_REPO, DWGitEvents.RESET_LOCAL_REPO, dwResetLocalRepo);
            CommandManager.register(DWGitEvents.CLONE_REPO, DWGitEvents.CLONE_REPO, dwCloneRepo);
            CommandManager.register(DWGitEvents.CREATE_TAG, DWGitEvents.CREATE_TAG, dwCreateTag);
            CommandManager.register(DWGitEvents.TEST_REMOTE, DWGitEvents.TEST_REMOTE, dwTestRemoteConnection);
        }
        //common events
        CommandManager.register(DWGitEvents.RELOAD_PREFERENCES, DWGitEvents.RELOAD_PREFERENCES, dwReloadPreferences);
        CommandManager.register(DWGitEvents.UPDATE_THEME, DWGitEvents.UPDATE_THEME, dwUpdateTheme);
        CommandManager.register(DWGitEvents.SET_APP_FONT, DWGitEvents.SET_APP_FONT, _addDynamicAppFont);
    }

    EventEmitter.on(Events.REFRESH_ALL, function () {
        if (window.dwGitPanelInstance === true) {
            Panel.refresh();
        } else if (window.dwGitDialogInstance === true) {
            updateDreamweaver(true);
        }
    });

    exports.init = init;
    exports.sendSingleFileGitStatusToDreamweaver = sendSingleFileGitStatusToDreamweaver;
    exports.getCurrentSelectedFilePath = getCurrentSelectedFilePath;
    exports.gitDialogShown = gitDialogShown;
    exports.updateDreamweaver = updateDreamweaver;
    exports.onCreateGitRepo = onCreateGitRepo;
    exports.sendGitStatusToDreamweaver = sendGitStatusToDreamweaver;
    exports.setGitRoot = dwSetGitRoot;
    exports.gitHistoryViewShown = gitHistoryViewShown;
    exports.onManageRemotes = onManageRemotes;
    exports.closeHistoryDialog = closeHistoryDialog;
    exports.loadAppTheme = loadAppTheme;
    exports.logEvent = logEvent;
});
